(*****************************************************************)
(************ Correspondence between two settings ****************)
(*****************************************************************)

(* Author: Nicolas Magaud *)
(* $Id: isomorphism.ml,v 1.6 2005/11/19 22:18:21 magaud Exp $ *)

open Pp;; (* msgnl *)
open Nameops;; (* pr_id *)
open Names;; (* Idmap *)
open Libobject;;

(*
exception Local of string;;
let failwith s = raise (Local(s));;
exception Not_found;;
*)

let iso_table = ref Idmap.empty;;

(* Synchronisation mechanism with reset *)

let _ = 
  let init () = iso_table := Idmap.empty in
  let freeze () = !iso_table in
  let unfreeze i = iso_table := i in
  Summary.declare_summary "iso-table"
    { Summary.freeze_function   = freeze;
      Summary.unfreeze_function = unfreeze;
      Summary.init_function     = init;
      Summary.survive_module    = true; (* for no good reason, just testing *)
      Summary.survive_section   = true }
;;

let cache_iso_table ((sp,_),(id1,id2)) =
  if Idmap.mem id1 !iso_table then
    (msgnl ((str "Error: ")++(pr_id id1)++
		 (str " is *already* translated into ")++
		 (pr_id (Idmap.find id1 !iso_table))++
		 (str ".")))
  else (iso_table := Idmap.add id1 id2 !iso_table;
    (msgnl ((pr_id id1)++
		 (str " is translated into ")++
		 (pr_id id2)++
		 (str "."))));;

let load_iso_table = fun _ -> cache_iso_table;;
let export_iso_table = function x -> Some x;;

(* Declaration of the iso-definition library object *)

let (in_iso_table,out_iso_table) =
  declare_object
    {  (default_object "ISO-DEF") with
       load_function = load_iso_table; 
       cache_function = cache_iso_table;
       export_function = export_iso_table
(*       classify_function = (fun (_,a) -> Keep(a))*)
};;

(* error handling ? duplicates are handled in cache_iso_table *)
let new_addiso id1 id2 =
  Lib.add_anonymous_leaf (in_iso_table (id1,id2));;

let check_iso id_c iso_t =
  if Idmap.mem id_c iso_t
  then
    let result = Idmap.find id_c iso_t in 
    msgnl ((pr_id id_c)++
	     (str " is translated into ")++
	     (pr_id result)++(str "."))
  else msgnl ((str "Error: isomorphism.ml: ")++(pr_id id_c)++
		(str " does *not* occur in the iso table."));;

let lookup_iso = Idmap.find;;

let show_iso () = 
  let nb_entries =  Idmap.fold (fun _ _ z -> z+1) !iso_table 0 in
  let header = 
    (str "Isomorphism so far: ")++
      (str (string_of_int nb_entries))++
      (str " entrie(s)") in
  (msgnl header;
   Idmap.iter 
     (fun id1 id2 -> msgnl ((pr_id id1)++
			      (str " is translated into ")++
			      (pr_id id2)++
			      (str "."))) !iso_table);;

VERNAC COMMAND EXTEND AddIso
  [ "AddIso" ident(i1) ident(i2) ]
   -> [ new_addiso i1 i2 ]
END


VERNAC COMMAND EXTEND CheckIso
  [ "CheckIso" ident(i1) ]
   -> [ check_iso i1 !iso_table]
END

VERNAC COMMAND EXTEND ShowIso
  [ "ShowIso" ]
   -> [ show_iso () ]
END

